import sys

# Povecavamo rekurziju za svaki slucaj, mada koristimo iterativni pristup gde možemo
sys.setrecursionlimit(200000)

class FenwickTree:
    def __init__(self, size):
        self.tree = [float('inf')] * (size + 1)
        self.map_idx = [-1] * (size + 1) # Cuva indeks tacke koja je dala minimum

    def update(self, i, val, pt_idx):
        # U ovom problemu tražimo minimum, pa update ide unazad (prema 1) ili unapred?
        # Kod 'sweep-line' za Manhattan MST tražimo min(x+y) za y >= yi.
        # Standardni Fenwick sumira [1, i]. Ovde nam treba query sufiksa ili prefiksa.
        # Koristicemo query na sufiksu (i do kraja) tako što mapiramo indekse inverzno
        # ili prosto idemo update ka "gore" a query ka "dole" za prefiks minimum.
        # Da ne komplikujemo: update na poziciji 'i', query minimuma od 'i' do kraja.
        # Standardni fenwick radi prefikse.
        # Ako hocemo query(index) -> min od index do kraja:
        # Update radi na 'index' i svim roditeljima naniže? Ne.
        # Radeci sa prefiksima (query(i) vraca min u [1, i]),
        # moramo da sortiramo ili transformišemo podatke tako da nam treba prefiks.
        
        # Ovde implementiramo standardni Fenwick za prefiks minimum.
        # update(i, val): postavi val na poziciju i, ažuriraj stablo
        i += 1 
        while i < len(self.tree):
            if val < self.tree[i]:
                self.tree[i] = val
                self.map_idx[i] = pt_idx
            i += i & (-i)

    def query(self, i):
        # Vraca (min_val, original_index) za opseg [0, i]
        i += 1
        res = float('inf')
        res_idx = -1
        while i > 0:
            if self.tree[i] < res:
                res = self.tree[i]
                res_idx = self.map_idx[i]
            i -= i & (-i)
        return res, res_idx
        
    def clear(self):
        self.tree = [float('inf')] * len(self.tree)
        self.map_idx = [-1] * len(self.tree)

class DisjointSet:
    def __init__(self, n):
        self.parent = list(range(n + 1))
    
    def find(self, i):
        path = []
        while i != self.parent[i]:
            path.append(i)
            i = self.parent[i]
        for node in path:
            self.parent[node] = i
        return i
    
    def union(self, i, j):
        root_i = self.find(i)
        root_j = self.find(j)
        if root_i != root_j:
            self.parent[root_i] = root_j
            return True
        return False

def solve():
    # Citanje ulaza
    input_data = sys.stdin.read().split()
    if not input_data:
        return

    iterator = iter(input_data)
    try:
        n = int(next(iterator))
        epsilon = float(next(iterator))
        points = []
        for i in range(n):
            x = float(next(iterator))
            y = float(next(iterator))
            points.append((x, y, i + 1)) # Cuvamo originalni indeks (1-based)
    except StopIteration:
        return

    # === Korak 1: Manhattan MST ===
    # Generisanje kandidata za grane (Manhattan edges)
    # Potrebno je naci najbližeg suseda u svakom oktantu.
    # Formula: dist(A, B) = |xA - xB| + |yA - yB|
    # Transformacije koordinata (4 prolaza) pokrivaju sve slucajeve.
    # Za xA <= xB i yA <= yB, dist = (xB + yB) - (xA + yA).
    # Maksimizujemo (xA + yA) tj. minimizujemo -(xA + yA) za tacke "levo-dole".
    # Ili jednostavnije, za fiksirano B, tražimo A takvo da je xA <= xB, yA <= yB
    # koje minimizuje (xB+yB) - (xA+yA). Pošto je B fiksno, maksimizujemo xA+yA.
    
    # Implementacija sweep-line algoritma za Manhattan MST
    edges = []
    
    # 4 transformacije: (x, y), (y, x), (-x, y), (-y, x) nije dovoljno,
    # treba pokriti 4 kvadranta oko svake tacke u odnosu na (x+y) i (x-y).
    # Standardni pristup: 
    # 1. Sortiraj po x. 
    # 2. Iteriraj i traži tacku sa y >= yi koja minimizuje (x+y) (tj. distanca).
    #    Pošto je x sortiran, uslov xi <= xj je zadovoljen ako idemo unazad ili pamtimo.
    
    # Zapravo, pravilna transformacija za 4 smera:
    # Za svaki od 4 smera:
    #   Sortiramo tacke po x.
    #   Za svaku tacku 'i', tražimo 'j' (j < i u sortiranom nizu) takvo da yj >= yi
    #   koje minimizuje (xi + yi) - (xj + yj) -> min ( -xj - yj ) -> max (xj + yj).
    #   Ovo radimo pomocu Fenwicka nad y-koordinatama.
    
    transformed_points = points[:]
    
    for _ in range(4):
        # Sortiramo po X
        transformed_points.sort(key=lambda p: (p[0], p[1]))
        
        # Kompresija koordinata za Y da bi stalo u Fenwick
        ys = sorted(list(set(p[1] for p in transformed_points)))
        y_map = {y: i for i, y in enumerate(ys)}
        ft_size = len(ys)
        ft = FenwickTree(ft_size)
        
        # Sweep-line s desna na levo
        # Tražimo j > i (u sortiranom nizu x raste, idemo unazad da bi x_j >= x_i)
        # Ali lakše je ici s leva na desno i tražiti u "prošlosti".
        # Tražimo j < i (znaci x_j <= x_i) takav da y_j >= y_i (uslov gornjeg kvadranta)
        # Minimizujemo |xi - xj| + |yi - yj|.
        # Pošto xj <= xi i pretpostavimo yj >= yi:
        # dist = (xi - xj) + (yj - yi) = (xi - yi) - (xj - yj)
        # Ovo znaci da tražimo max(xj - yj) za yj >= yi.
        
        # Fenwick cuva max(x - y) na pozicijama mapiranim po y.
        # Pošto standardni Fenwick radi prefikse, a nama treba yj >= yi (sufiks),
        # mapiracemo y inverzno: rank = len - rank.
        
        for x, y, idx in transformed_points:
            r = y_map[y]
            # yj >= yi -> u inverznom mapiranju, rank_j <= rank_i.
            # Znaci query(inverz_rank(y)) nam daje max(x-y) za y_j >= y_i
            inv_r = ft_size - 1 - r
            
            val, best_idx = ft.query(inv_r)
            if best_idx != -1:
                # val je -min_val, jer Fenwick inicijalno implementiran za min
                # Cekaj, klasa gore je za min. Prilagodicemo logiku klasi.
                # Želimo min (xi - yi) + (-xj + yj).
                # Znaci u strukturi cuvamo (-xj + yj).
                dist = (x - y) + val
                edges.append((dist, idx, best_idx))
            
            # Dodajemo trenutnu tacku: cuvamo -x + y
            ft.update(inv_r, -x + y, idx)

        # Rotacija tacaka za 90 stepeni: (x, y) -> (y, -x)
        # Ali pošto radimo sa Manhattan distancom koja je simetricna po osama,
        # dovoljno je swapovati x i y. I moramo paziti na refleksije.
        # Najsigurniji nacin za 4 kvadranta:
        # 1. (x, y)
        # 2. (y, x) -> Swap x,y. Pokriva simetriju.
        # 3. (-x, y) -> Refleksija.
        # 4. (-y, x) -> Swap i refleksija.
        
        # Zapravo, u petlji samo transformišemo koordinate za sledecu iteraciju.
        # (x, y) -> (y, x) nije dovoljno jer samo menja osu.
        # Treba nam rotacija: x' = y, y' = -x. 
        # Zbog jednostavnosti sa pozitivnim brojevima i sortom: x' = y, y' = 1 - x (pošto je unit square).
        
        next_points = []
        for x, y, idx in transformed_points:
            next_points.append((y, 1.0 - x, idx)) # Rotacija 90 stepeni
        transformed_points = next_points

    # Sada imamo kandidate za MST. Sortiramo grane.
    edges.sort(key=lambda x: x[0])
    
    # Kruskalov algoritam za Originalni MST
    ds_full = DisjointSet(n)
    mst_adj = [] # Lista grana u originalnom MST-u: (u, v)
    
    edges_count = 0
    for w, u, v in edges:
        if ds_full.union(u, v):
            mst_adj.append((u, v))
            edges_count += 1
            if edges_count == n - 1:
                break
                
    # === Korak 2: Grid Clustering ===
    # Cilj: |Q| <= 18 / epsilon^2.
    # Grid cell size d. (1/d)^2 <= 18/eps^2 => d >= eps / sqrt(18) ~ eps / 4.24
    # Uzimamo step = epsilon / 4.0 da budemo sigurni.
    
    step = epsilon / 4.0
    # Da izbegnemo deljenje nulom u granicnim slucajevima ili preveliki broj
    if step == 0: step = 1e-9 # Ne bi smelo zbog ogranicenja
    
    grid_reps = {} # (grid_x, grid_y) -> original_index
    point_to_rep = {} # original_index -> rep_index
    
    # Originalne koordinate nam trebaju za mapiranje
    # points lista nije menjana, sadrži (x, y, idx)
    # Ali points je bio nesortiran pre prve rotacije? Ne, rotirali smo kopiju.
    # Originalni 'points' je u redosledu unosa? Da.
    # Ali 'points' sadrži tuple. Mapiramo po indeksu.
    # Da bi bilo brže, napravimo listu gde je index -> (x, y)
    coords = [None] * (n + 1)
    for x, y, idx in points:
        coords[idx] = (x, y)
        
    for i in range(1, n + 1):
        x, y = coords[i]
        gx = int(x / step)
        gy = int(y / step)
        if (gx, gy) not in grid_reps:
            grid_reps[(gx, gy)] = i
        
        rep = grid_reps[(gx, gy)]
        point_to_rep[i] = rep
        
    # Skup Q su svi values iz grid_reps
    Q_indices = sorted(list(set(grid_reps.values())))
    
    # === Korak 3: Konstrukcija stabla T ===
    # Koristimo strukturu originalnog MST-a
    ds_final = DisjointSet(n) # Union-find nad svim tackama, ali nas zanimaju samo repovi
    final_edges = []
    
    for u, v in mst_adj:
        rep_u = point_to_rep[u]
        rep_v = point_to_rep[v]
        
        if rep_u != rep_v:
            # Pokušaj da povežeš predstavnike
            if ds_final.union(rep_u, rep_v):
                final_edges.append((rep_u, rep_v))
                
    # Ispis rešenja
    print(len(Q_indices))
    print(*(Q_indices))
    for u, v in final_edges:
        print(f"{u} {v}")

if __name__ == '__main__':
    solve()